import { json } from '@sveltejs/kit';
import bcrypt from 'bcrypt';
import { db } from '$lib/server/db.js';

/** @type {import('./$types').RequestHandler} */
export async function GET({ params }) {
  try {
    const userId = parseInt(params.id);

    if (!userId || isNaN(userId)) {
      return json(
        { error: 'User ID tidak valid' },
        { status: 400 }
      );
    }

    const [users] = await db.query(
      'SELECT id, fullName, email, phone, role, major, status, createdAt FROM users WHERE id = ?',
      [userId]
    );

    if (users.length === 0) {
      return json(
        { error: 'User tidak ditemukan' },
        { status: 404 }
      );
    }

    return json({
      success: true,
      user: users[0]
    });

  } catch (error) {
    console.error('Get user API error:', error);
    return json(
      { error: 'Terjadi kesalahan pada server' },
      { status: 500 }
    );
  }
}

/** @type {import('./$types').RequestHandler} */
export async function PATCH({ params, request }) {
  try {
    const userId = parseInt(params.id);
    if (!userId || isNaN(userId)) {
      return json({ error: 'User ID tidak valid' }, { status: 400 });
    }

    const body = await request.json();
    const { password, currentPassword } = body;
    if (!password || typeof password !== 'string' || password.trim().length < 6) {
      return json(
        { error: 'Password baru minimal 6 karakter' },
        { status: 400 }
      );
    }

    const [users] = await db.query('SELECT id, password FROM users WHERE id = ?', [userId]);
    if (users.length === 0) {
      return json({ error: 'User tidak ditemukan' }, { status: 404 });
    }

    const user = users[0];
    // Jika dikirim currentPassword (ganti password oleh user), wajib verifikasi dulu
    if (currentPassword != null && String(currentPassword).trim() !== '') {
      const stored = user.password;
      const isBcrypt = stored && (stored.startsWith('$2a$') || stored.startsWith('$2b$') || stored.startsWith('$2y$'));
      if (!isBcrypt || !(await bcrypt.compare(String(currentPassword).trim(), stored))) {
        return json({ error: 'Password saat ini salah' }, { status: 401 });
      }
    }

    const hashedPassword = await bcrypt.hash(password.trim(), 10);
    await db.query('UPDATE users SET password = ? WHERE id = ?', [hashedPassword, userId]);

    return json({ success: true, message: 'Password berhasil diperbarui' });
  } catch (error) {
    console.error('Update password API error:', error);
    return json(
      { error: 'Terjadi kesalahan pada server' },
      { status: 500 }
    );
  }
}

